<?php

namespace App\Livewire\Actions\Billings;

use App\Traits\Billing;
use App\Models\Websystem;
use Illuminate\Support\Str;
use App\Traits\GenerateName;
use Illuminate\Support\Carbon;
use App\Models\Billings\Invoice;
use App\Traits\CustomerPaketTrait;
use Illuminate\Support\Facades\Log;
use App\Services\GeneralLogServices;
use Illuminate\Support\Facades\Auth;
use App\Services\Billings\TaxService;
use App\Models\Customers\CustomerPaket;
use Illuminate\Support\Facades\Storage;
use App\Services\Billings\PricingService;
use App\Services\Billings\DeadlineService;
use App\Traits\WebSystemTrait;

class InvoiceAction
{
    use Billing;
    use GenerateName;
    use CustomerPaketTrait, WebSystemTrait;

    protected $pricingService;
    protected $deadlineService;
    protected $generalLogServices;
    public function __construct(PricingService $pricingService = null, DeadlineService $deadlineService = null, GeneralLogServices $generalLogServices = null)
    {
        $this->pricingService = $pricingService ?? new PricingService();
        $this->deadlineService = $deadlineService ?? new DeadlineService();
        $this->generalLogServices = $generalLogServices ?? new GeneralLogServices();
    }

    /**
     * Add customer paket
     *
     * @param  array<string, string>  $input
     */
    public function create_invoice(CustomerPaket $customerPaket, $invoiceNumber): Invoice
    {
        $intervalInvoiceDay = $this->different_day_create_billing();
        $renewalPeriod = $this->getRenewalPeriod($customerPaket->renewal_period);

        $mikrotikAutoIsolir = $customerPaket->mikrotik->auto_isolir;
        if ($mikrotikAutoIsolir->activation_date) {
            if ($this->isPrabayar()) {
                // dd($customerPaket->expired_date);
                $startInvoicePeriod = Carbon::parse(is_null($customerPaket->expired_date) ? Carbon::now() : $customerPaket->expired_date);
                $endInvoicePeriod = Carbon::parse($startInvoicePeriod)->add($renewalPeriod);
                $nextBilledAt = Carbon::parse($endInvoicePeriod)->subDays($intervalInvoiceDay);
                $dueDate = $startInvoicePeriod;
            } else {
                $startInvoicePeriod = Carbon::parse($customerPaket->start_date);
                $endInvoicePeriod = Carbon::parse($startInvoicePeriod)->add($renewalPeriod);
                $nextBilledAt = Carbon::parse($customerPaket->expired_date)->subDays($intervalInvoiceDay)->add($renewalPeriod);
                $dueDate = $endInvoicePeriod;
            }
        } else {
            if ($this->isPrabayar()) {
                $startInvoicePeriod = Carbon::parse(is_null($customerPaket->expired_date) ? Carbon::now()->setDay((int)$mikrotikAutoIsolir->due_date) : $customerPaket->expired_date);
                $endInvoicePeriod = Carbon::parse($startInvoicePeriod)->add($renewalPeriod);
                $nextBilledAt = Carbon::parse($endInvoicePeriod)->subDays($intervalInvoiceDay);
                $dueDate = $startInvoicePeriod;
            } else {
                $startInvoicePeriod = Carbon::parse(is_null($customerPaket->start_date) ? Carbon::now()->setDay((int)$mikrotikAutoIsolir->due_date) : $customerPaket->start_date);
                $endInvoicePeriod = Carbon::parse($startInvoicePeriod)->add($renewalPeriod);
                $nextBilledAt = Carbon::parse($customerPaket->expired_date)->subDays($intervalInvoiceDay)->add($renewalPeriod);
                $dueDate = $endInvoicePeriod;
            }
        }

        $invoicePeriod = Carbon::parse($startInvoicePeriod)->startOfMonth();

        $invoice = Invoice::create([
            'user_customer_id' => $customerPaket->user->user_customer->id,
            'customer_paket_id' => $customerPaket->id,
            'periode' => $invoicePeriod,
            'invoice_number' => $invoiceNumber,
            'issue_date' => now(),
            'due_date' => $dueDate,
            'start_periode' => $startInvoicePeriod,
            'end_periode' =>  $endInvoicePeriod,
            'status' => 'pending',
        ]);

        $customerPaket->forceFill([
            'next_billed_at' => $nextBilledAt->startOfDay()
        ])->save();
        $this->generalLogServices->create_invoice($customerPaket, $invoicePeriod, Auth::user() ? Auth::user()->full_name : 'system');

        return  $invoice;
    }

    public function add_discount(Invoice $invoice, $discount)
    {
        try {
            $totalDiscount = $invoice->discount + $discount;
            $tax = (new TaxService)->calculateTax($invoice->amount - $totalDiscount);

            $invoice->update([
                'discount' => $discount,
                'tax' => $tax
            ]);
            return [
                'success' => true,
                'data' => $invoice
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    public function sub_discount(Invoice $invoice, $discount)
    {
        try {
            $discount = $invoice->discount - $discount;
            $tax = (new TaxService)->calculateTax($invoice->amount - $discount);

            $invoice->update([
                'discount' => $discount,
                'tax' => $tax
            ]);
            return [
                'success' => true,
                'data' => $invoice
            ];
        } catch (\Exception $e) {
            return [
                'success' => false,
                'message' => $e->getMessage()
            ];
        }
    }

    public function delete_billing(Invoice $invoice)
    {
        try {
          //  $this->delete_invoice_file($invoice->invoice_path);
            $invoice->forceDelete();
            return true;
        } catch (\Exception $e) {
            return $e;
        }
    }

    /*
    public function delete_invoice_file($invoicePath)
    {
        if (Storage::disk('local')->exists($invoicePath)) {
            Storage::disk('local')->delete($invoicePath);
        }
    }
        */
}
