<?php

namespace App\Livewire\Admin\Billings\Modal;

use App\Models\Bank;
use Livewire\Component;
use Livewire\Attributes\On;
use App\Models\Billings\Order;
use App\Models\PaymentGateway;
use App\Traits\WebSystemTrait;
use Illuminate\Support\Carbon;
use App\Models\Billings\Invoice;
use App\Traits\NotificationTrait;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Http\Requests\Billings\PaymentRequest;
use App\Models\Customers\CustomerPaketAddress;
use App\Services\Payments\PartialPaymentService;
use App\Services\Payments\PaymentGatewayService;

class BillingPayment extends Component
{
    use WebSystemTrait, NotificationTrait;
    public $input = [];
    public Invoice $invoice;
    public $expiredDate;
    public $paymentModal = false;
    public $account_bank;
    private $paymentGatewayService;
    public $paymentChanels = [];
    public $order;
    public function __construct()
    {
        $this->paymentGatewayService = new PaymentGatewayService;
    }

    #[On('billing-payment-modal')]
    public function showPaymentModal($invoice)
    {
        $this->reset();
        if (Auth::user()->can('invoice payment') || Auth::user()->hasRole('admin')) {
            $invoice = Invoice::findOrFail($invoice);
            $this->invoice = $invoice;
            $this->input['send_notification'] = $invoice->customer_paket->customer_billing_address->wa_notification ? true : false;

            $this->paymentModal = true;
            $customerPaket = $this->invoice->customer_paket;
            $expiredDate = Carbon::parse($customerPaket->expired_date);
            $paylaterDate = Carbon::parse($customerPaket->paylater_date);
            $this->expiredDate = $customerPaket->paylater_date ? ($expiredDate->gt($paylaterDate) ? $expiredDate : $paylaterDate) : $expiredDate;
        } else {
            $this->error_notification('Unauthorized', 'You dont have permission.');
        }
    }

    public function updatedInputSelectedPaymentMethode($paymentMethode)
    {
        $paymentMethode && $paymentMethode != 'paylater' ? $this->dispatch('input-amount', open: true) : $this->dispatch('input-amount', open: false);
        if ($paymentMethode === 'paylater') {
            $this->input['paylaterDate'] = $this->expiredDate->format('Y-m-d');
            $this->dispatch('select-paylater-date', open: true);
        } else {
            $this->dispatch('select-paylater-date', open: false);
            $this->input['amount'] = $this->invoice->remaining_amount;
            $this->input['paylaterDate'] = null;
        }

        if ($paymentMethode === 'bank_transfer') {
            $this->dispatch('select-methode-transfer', open: true);
        } else {
            $this->dispatch('select-methode-transfer', open: false);
        }

        if ($paymentMethode === 'tripay') {
            $this->requestPaymentChannel();
            $order = $this->invoice->order;
            if ($order && $order->status == 'pending' && Carbon::parse($order->expired_time)->gte(Carbon::now())) {
                $this->order = $order;
                $this->input['amount'] = $order->amount;
            }
        } else {
            $this->dispatch('select-methode-tripay', open: false);
        }
    }

    public function updatedInputAmount($value)
    {
        if ($value < $this->invoice->remaining_amount) {
            $this->dispatch('select-paylater-date', open: true);
            $this->input['paylaterDate'] = $this->expiredDate->format('Y-m-d');
        } else {
            $this->input['paylaterDate'] = null;
            $this->dispatch('select-paylater-date', open: false);
        }
    }

    public function payment(PaymentRequest $request, PartialPaymentService $partialPaymentService)
    {
        $request->validate($this->input, $this->invoice->remaining_amount);
        if ($this->input['selectedPaymentMethode'] === 'tripay' || $this->input['selectedPaymentMethode'] === 'midtrans') {
            $this->processOrder();
        } elseif ($this->input['selectedPaymentMethode'] === 'paylater') {
            $this->invoice->customer_paket->forceFill([
                'paylater_date' => $this->input['paylaterDate']
            ])->save();
            $this->dispatch('refresh-billing-paket');
            $message = trans('billing.alert.payment-success-message', ['periode' => Carbon::parse($this->invoice->periode)->format('F Y'), 'paket' => $this->invoice->customer_paket->paket->name, 'customer' =>  $this->invoice->customer_paket->user->first_name]);
            $this->success_notification(trans('billing.alert.payment-success'), $message);
            $this->closeModal();
        } else {
            $bank = null;
            if ($this->input['selectedPaymentMethode'] === 'bank_transfer') {
                $bank = Bank::whereSlug($this->input['selectedBankTransfer'])->first();
                $bank = $bank->bank_name . ' - ' . $bank->account_name . ' - ' . $bank->account_number;
            }

            $paymentResult = $partialPaymentService->processPartialPayment(
                $this->invoice,
                $this->input['amount'] == '' ? 0 : $this->input['amount'],
                $this->input['selectedPaymentMethode'],
                $bank,
                Auth::user()->full_name
            );

            if ($paymentResult['success']) {
                if ($this->input['paylaterDate']) {
                    $this->invoice->customer_paket->forceFill([
                        'paylater_date' => $this->input['paylaterDate']
                    ])->save();
                }
                $this->invoice->forceFill([
                    'notes' => $this->input['note'] ?? null
                ])->save();

                $payment = $paymentResult['payment'];
                $payment->forceFill(['notes' => $this->input['note'] ?? null])->save();
                //Update invoice status and mikrotik on InvoiceObserver
                $this->dispatch('refresh-billing-paket');
                $message = trans('billing.alert.payment-success-message', ['periode' => Carbon::parse($this->invoice->periode)->format('F Y'), 'paket' => $this->invoice->customer_paket->paket->name, 'customer' =>  $this->invoice->customer_paket->user->first_name]);
                $this->success_notification(trans('billing.alert.payment-success'), $message);

                $this->closeModal();
            } else {
                $this->error_notification(trans('billing.alert.payment-failed'), $paymentResult['message']);
            }
        }
    }

    public function updatedInputMethod($methode)
    {

        if ($methode === 'DANA') {
            $this->dispatch('input-customer-billing-phone', open: true);
            $this->input['customerPhone'] = $this->invoice->customer_paket->customer_billing_address->phone;
        } else {
            $this->dispatch('input-customer-billing-phone', open: false);
        }
    }

    private function processOrder()
    {
        Validator::make($this->input, [
            'method' => ['required'],
            'customerPhone' => ['required_if:method,==,DANA', 'nullable'],
        ])->validate();

        $res = $this->paymentGatewayService->createTransaction($this->invoice, $this->input['method']);
        if ($res['success'] && $res['data']['status'] == 'UNPAID') {
            if ($this->invoice->orders->count()) $this->invoice->orders()->forceDelete();
            $this->invoice->forceFill(['status' => 'process'])->save();
            $res['data']['status'] = 'pending';
            unset($res['data']['checkout_url']);
            unset($res['data']['callback_url']);
            unset($res['data']['return_url']);
            $res['data']['expired_time'] = Carbon::parse($res['data']['expired_time'])->setTimezone('Asia/Jakarta');
            $res['data']['order_items'] = json_encode($res['data']['order_items']);
            $res['data']['instructions'] = json_encode($res['data']['instructions']);
            $res['data']['payment_gateway_channel'] = PaymentGateway::whereIsActive(true)->first()->value;

            $this->order = new Order($res['data']);
            $this->order = $this->invoice->order()->save($this->order);
        } else {
            $title = trans('Gagal');
            $message = $res['message'] ?? trans('payment-gateway.message.failed-status');
            $this->error_notification($title, $message);
        }
    }

    private function requestPaymentChannel()
    {
        $this->paymentChanels = $this->paymentGatewayService->requestPaymentChanel();
        if ($this->paymentChanels['payment_chanels']['success']) {
            $this->dispatch('select-methode-tripay', open: true);
            $this->paymentChanels = $this->paymentChanels['payment_chanels']['data'];
        } else {
            Log::info('Tripay: ' . $this->paymentChanels['payment_chanels']['message']);
            $title = 'Failed';
            $message =  'Tripay: ' . $this->paymentChanels['payment_chanels']['message'];
            $this->error_notification($title, $message);
        }
    }

    public function changePaymentMethod()
    {
        $this->order->delete();
        $this->order = null;
        $this->processOrder();
    }

    public function closeModal()
    {
        $this->paymentModal = false;
    }

    public function render()
    {
        return view('livewire.admin.billings.modal.billing-payment');
    }
}
