<?php

namespace App\Livewire\Admin\Billings\Modal;

use App\Traits\NotificationTrait;
use Livewire\Component;
use Livewire\Attributes\On;
use App\Models\Billings\Payment;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Validator;
use App\Services\Payments\RefundPaymentService;

class BillingRefund extends Component
{
    use NotificationTrait;
    public $input = [];
    public Payment $payment;
    public $refundModal = false;

    #[On('billing-refund-modal')]
    public function showBillingUnpaymentModal(Payment $payment)
    {
        if (Auth::user()->can('invoice payment') || Auth::user()->hasRole('admin')) {
            $this->payment = $payment;
            $this->input['amount'] = $payment->remaining_refundable_amount;
            $this->refundModal = true;
        } else {
            $this->error_notification('Unauthorized', 'You dont have permission.');
        }
    }

    public function unpayment(RefundPaymentService $refundPaymentService)
    {
        Validator::make($this->input, [
            'current_password' => ['required', 'string', 'current_password:web'],
            'amount' => ['required', 'lte:' . $this->payment->amount],
        ], [
            'amount.lte' => __('Refund amount cannot exceed the original payment amount (' . number_format($this->payment->amount, 0) . ').'),
            'current_password.required' => __('customer.validation-error-message.current-password-required'),
        ])->validate();

        $response = $refundPaymentService->processRefund(
            $this->payment,
            $this->input['amount']
        );

        $this->dispatch('refresh-payment-list');
        $this->refundModal = false;
        if ($response['success']) {
            $message = trans('billing.alert.unpayment-success-message', ['paket' => $this->payment->invoice->customer_paket->paket->name, 'customer' =>  $this->payment->invoice->customer_paket->user->full_name]);
            $this->success_notification(trans('billing.alert.unpayment-success-title'), $message);
        } else {
            $message =  $response['message'];
            $this->error_notification(trans('billing.alert.unpayment-failed-title'), $message);
        }
    }

    public function render()
    {
        return view('livewire.admin.billings.modal.billing-refund');
    }
}
