<?php

namespace App\Livewire\Admin\Customers\Modal;

use App\Models\User;
use Livewire\Component;
use Livewire\Attributes\On;
use App\Traits\NotificationTrait;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Services\GeneralLogServices;
use Illuminate\Support\Facades\Auth;
use App\Http\Requests\CurrentPasswordRequest;

class ConfirmBulkDeletePermanently extends Component
{
    use NotificationTrait;
    public $bulkDeleteCustomerPermanentlyModal = false;
    public $usersDeleted = [];
    public $input = [];


    #[On('bulk-delete-permanent-customer-modal')]
    public function showBulkDeleteCustomerPermanentlyModal($usersSelected)
    {
        if (Auth::user()->can('customer action') || Auth::user()->hasRole('admin')) {
            $this->bulkDeleteCustomerPermanentlyModal = true;
            $usersDeleted = User::query()
                ->whereIn('id', $usersSelected)
                ->withTrashed()
                ->get();
            $this->usersDeleted = $usersDeleted;
        } else {
            $this->error_notification('Unauthorized', 'You dont have permission.');
        }
    }

    public function bulkDeleteCustomerPermanently(CurrentPasswordRequest $currentPasswordRequest, GeneralLogServices $generalLogServices)
    {
        $this->resetErrorBag();
        $currentPasswordRequest->validate($this->input);

        $successDelete = 0;
        $failedDelete = 0;
        foreach ($this->usersDeleted as $userDeleted) {
            DB::beginTransaction();
            try {
                $customerName = $userDeleted->full_name;
                $userDeleted->forceDelete();
                //Add log
                $generalLogServices->admin_action($generalLogServices::DELETE_CUSTOMER_PERMANENTLY, "Delete permanently " . $customerName, Auth::user()->full_name);
                $successDelete++;
                DB::commit();
            } catch (\Exception $e) {
                $failedDelete++;
                DB::rollBack();
                Log::info($e->getMessage());
            }
        }

        $this->success_notification(trans('customer.alert.delete'), trans('customer.alert.customer-bulk-delete-detail', ['countSuccess' => $successDelete, 'countFailed' => $failedDelete]));
        $this->closeModal();
    }

    public function closeModal()
    {
        $this->bulkDeleteCustomerPermanentlyModal = false;
        $this->dispatch('refresh-customer-list');
        $this->dispatch('refresh-selected-users');
    }

    public function render()
    {
        return view('livewire.admin.customers.modal.confirm-bulk-delete-permanently');
    }
}
