<?php

namespace App\Livewire\Admin\Customers\Modal;

use App\Models\User;
use Livewire\Component;
use Livewire\Attributes\On;
use App\Services\CustomerService;
use App\Traits\NotificationTrait;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use App\Services\GeneralLogServices;
use Illuminate\Support\Facades\Auth;
use App\Services\CustomerPaketService;
use App\Http\Requests\CurrentPasswordRequest;

class ConfirmBulkRestoreCustomer extends Component
{
    use NotificationTrait;
    public $confirmBulkRestoreCustomerModal = false;
    public $usersDeleted = [];
    public $input = [];


    #[On('bulk-restore-customer-deleted-modal')]
    public function showConfirmBulkRestoreCustomerModal($usersSelected)
    {
        if (Auth::user()->can('customer action') || Auth::user()->hasRole('admin')) {
            $this->confirmBulkRestoreCustomerModal = true;
            $this->input['restoreOnMikrotik'] = true;
            $usersDeleted = User::query()
                ->whereIn('id', $usersSelected)
                ->withTrashed()
                ->get();
            $this->usersDeleted = $usersDeleted;
        } else {
            $this->error_notification('Unauthorized', 'You dont have permission.');
        }
    }

    public function bulkRestoreCustomer(CurrentPasswordRequest $currentPasswordRequest, CustomerService $customerService, GeneralLogServices $generalLogServices)
    {
        $this->resetErrorBag();
        $currentPasswordRequest->validate($this->input);

        $successRestore = 0;
        $failedRestore = 0;
        foreach ($this->usersDeleted as $userDeleted) {
            DB::beginTransaction();
            try {
                $customerService->restoreCustomer($userDeleted, $this->input['restoreOnMikrotik']);
                $generalLogServices->admin_action($generalLogServices::RESTORE_CUSTOMER, "Restore customer " . $userDeleted->full_name);
                $successRestore++;
                DB::commit();
            } catch (\Exception $e) {
                $failedRestore++;
                DB::rollBack();
                Log::info($e->getMessage());
            }
        }

        $this->success_notification(trans('customer.alert.notif-bulk-customer-restore-title'), trans('customer.alert.notif-bulk-restore-customer-detail', ['countSuccess' => $successRestore, 'countFailed' => $failedRestore]));
        $this->closeModal();
    }

    public function closeModal()
    {
        $this->confirmBulkRestoreCustomerModal = false;
        $this->dispatch('refresh-deleted-customer-list');
        $this->dispatch('refresh-selected-users');
    }

    public function render()
    {
        return view('livewire.admin.customers.modal.confirm-bulk-restore-customer');
    }
}
