<?php

namespace App\Livewire\Admin\Customers\Modal;

use Flux\Flux;
use App\Models\Address;
use Livewire\Component;
use Livewire\Attributes\On;
use App\Services\User\UserService;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Auth;
use App\Traits\AddressCloudAlertTrait;
use App\Http\Requests\Customers\CreateCustomerStepOneRequest;
use App\Http\Requests\Customers\CreateCustomerStepTwoRequest;
use App\Http\Requests\Customers\CreateCustomerStepThreeRequest;
use App\Traits\NotificationTrait;

class CreateNewCustomerModal extends Component
{
    use AddressCloudAlertTrait, NotificationTrait;
    public $input = [];
    public $addCustomerModal = false;
    public $currentStep = 1;

    public $country, $province, $city, $district, $subdistrict;

    /**
     * Add or Edit User Modal
     */
    #[On('add-customer-modal')]
    public function showAddCustomerModal()
    {
        if (Auth::user()->can('customer action') || Auth::user()->hasRole('admin')) {
            $this->addCustomerModal = true;
        } else {
            $this->error_notification('Unauthorized', 'You dont have permission.');
        }
    }

    public function firstStepSubmit()
    {
        (new CreateCustomerStepOneRequest())->validate($this->input);
        $this->currentStep = 2;
    }

    public function secondStepSubmit()
    {
        $this->input = array_merge($this->input, [
            'country' => $this->country->name ?? '',
            'province' => $this->province->name ?? '',
            'city' => $this->city->name ?? '',
            'district' => $this->district->name ?? '',
            'subdistrict' => $this->subdistrict->name ?? ''
        ]);
        $this->input['role'] = 'customer';
        (new CreateCustomerStepTwoRequest())->validate($this->input);
        $this->currentStep = 3;
    }

    protected $listeners = [
        'countryUpdated' => 'setCountry',
        'provinceUpdated' => 'setProvince',
        'cityUpdated' => 'setCity',
        'districtUpdated' => 'setDistrict',
        'subdistrictUpdated' => 'setSubdistrict',
    ];
    public function setCountry($object)
    {
        $this->country = Address::find($object['value']);
    }
    public function setProvince($object)
    {
        $this->province = Address::find($object['value']);
    }
    public function setCity($object)
    {
        $this->city = Address::find($object['value']);
    }
    public function setDistrict($object)
    {
        $this->district = Address::find($object['value']);
    }
    public function setSubdistrict($object)
    {
        $this->subdistrict = Address::find($object['value']);
    }

    public function addCustomer()
    {
        (new CreateCustomerStepThreeRequest())->validate($this->input);
        DB::beginTransaction();
        $user =  (new UserService())->addUserCustomer($this->input);
        DB::commit();

        $this->dispatch('refresh-customer-list');
        $this->success_notification(trans('customer.alert.customer-created'), trans('customer.alert.customer-created-successfully', ['user' => $user->first_name]));
        $this->closeAddCustomerModal();
    }

    public function back($step)
    {
        $this->currentStep = $step;
    }

    /**
     * Close add user modal
     */
    #[On('close-add-customer-modal')]
    public function closeAddCustomerModal()
    {
        $this->currentStep = 0;
        Flux::modal('add-customer-modal')->close();
        $this->addCustomerModal = false;
        $this->reset();
    }

    public function render()
    {
        return view('livewire.admin.customers.modal.create-new-customer-modal');
    }
}
