<?php

namespace App\Services\Payments;


use App\Traits\Billing;
use App\Models\Websystem;
use App\Traits\GenerateName;
use App\Traits\WebSystemTrait;
use Illuminate\Support\Carbon;
use App\Models\Billings\Payment;
use Illuminate\Support\Facades\Log;
use App\Models\Customers\AutoIsolir;
use App\Services\CustomerPaketService;
use App\Services\Mikrotiks\MikrotikService;
use App\Services\Mikrotiks\MikrotikPppService;
use App\Services\Mikrotiks\MikrotikIpStaticService;

class MikrotikPaymentService
{
    use GenerateName, Billing, WebSystemTrait;

    private MikrotikPppService $pppService;
    private MikrotikIpStaticService $ipStaticService;
    private MikrotikService $mikrotikService;
    private CustomerPaketService $customerPaketService;
    public function __construct()
    {
        $this->pppService = new MikrotikPppService;
        $this->ipStaticService = new MikrotikIpStaticService;
        $this->mikrotikService = new MikrotikService;
        $this->customerPaketService = new CustomerPaketService;
    }

    /**
     * Summary of mikrotik_unpayment_process
     * @param mixed $invoice
     * @return array{message: mixed, success: bool|array{message: string, success: bool}|array{success: bool}}
     */
    //public function mikrotik_unpayment_process($customerPaket)
    public function mikrotik_payment_process($customerPaket)
    {
        switch ($customerPaket->internet_service->value) {
            case 'ppp':
                return $this->update_payment_secret_ppp_on_mikrotik($customerPaket);
            case 'ip_static':
                return $this->update_payment_ip_static_on_mikrotik($customerPaket);
            default:
                return [
                    'success' => false,
                    'message' =>  'Unsupported internet service.'
                ];
        }
    }

    public function update_payment_secret_ppp_on_mikrotik($customerPaket)
    {
        $mikrotik = $customerPaket->paket->mikrotik;
        $customerPppPaket = $customerPaket->customer_ppp_paket;
        $userSecretResponse = $this->pppService->getPppSecret($mikrotik, $customerPppPaket);
        if ($userSecretResponse['success']) {
            $expiredDatePaket = Carbon::parse($customerPaket->expired_date);
            $paylaterDate = Carbon::parse($customerPaket->paylater_date);
            $expiredDate = $customerPaket->paylater_date ? ($expiredDatePaket->gt($paylaterDate) ? $expiredDatePaket : $paylaterDate) : $expiredDatePaket;

            $comment = $this->commentMikrotik($expiredDate);

            if ($expiredDate->isPast()) {
                $profileIsolir = $mikrotik->auto_isolir->profile_id;
                if ($this->autoIsolirDriverMikrotik() && !$mikrotik->auto_isolir->disabled) {
                    return $this->pppService->updateProfileSecret($mikrotik, $customerPppPaket, $profileIsolir, $comment);
                } else {
                    return $this->pppService->updateCommentSecret($mikrotik, $customerPppPaket, $comment);
                }
            } else {
                return $this->pppService->updateCommentSecret($mikrotik, $customerPppPaket, $comment);
            }
        }
        return $userSecretResponse;
    }

    private function update_payment_ip_static_on_mikrotik($customerPaket)
    {
        $mikrotik = $customerPaket->paket->mikrotik;
        $ipAddress = $customerPaket->customer_static_paket->ip_address;

        $expiredDate = Carbon::parse($customerPaket->expired_date);
        $paylaterDate = Carbon::parse($customerPaket->paylater_date);
        $expiredDate = $customerPaket->paylater_date ? ($expiredDate->gt($paylaterDate) ? $expiredDate : $paylaterDate) : $expiredDate;
        $comment = $this->commentMikrotik($expiredDate);
        if ($expiredDate->isPast()) {
            $this->ipStaticService->addIpToAddressList($mikrotik, $ipAddress, $mikrotik->auto_isolir->address_list_isolir);
        }
        $this->ipStaticService->updateCommentSimpleQueue($mikrotik, $ipAddress, $comment);
        return $this->ipStaticService->updateCommentArp($mikrotik, $ipAddress, $comment);
    }

    private function commentMikrotik($date)
    {
        $commentUnpayment = $this->websystem_comment_unpayment();
        return $commentUnpayment . '_' . Carbon::parse($date)->format('d_m_Y');
    }
}
