<?php

namespace App\Services\Payments;

use Exception;
use App\Models\Billings\Invoice;
use App\Models\Billings\Payment;
use Illuminate\Support\Facades\Log;
use App\Models\Billings\PaymentHistory;
use Illuminate\Support\Carbon;

class PaymentReconciliationService
{
    public function reconcilePayment(Payment $payment)
    {
        try {
            // Try to match payment with invoice
            $invoice = $this->findMatchingInvoice($payment);

            if ($invoice) {
                return $this->processReconciliation($payment, $invoice);
            }

            // Mark as unreconciled if no match found
            $payment->forceFill([
                'reconciliation_status' => 'unmatched',
                'reconciliation_notes' => 'No matching invoice found'
            ]);

            $this->logReconciliationHistory($payment, null, 'unmatched');

            return false;
        } catch (Exception $e) {
            Log::error('Payment reconciliation failed', [
                'payment_id' => $payment->id,
                'error' => $e->getMessage()
            ]);

            $payment->update([
                'reconciliation_status' => 'failed',
                'reconciliation_notes' => $e->getMessage()
            ]);

            return false;
        }
    }

    protected function findMatchingInvoice(Payment $payment)
    {
        // Try to match by invoice_id if available
        if ($payment->invoice_id) {
            return Invoice::find($payment->invoice_id);
        }

        // Try to match by amount and customer
        return Invoice::where('user_customer_id', $payment->user_customer_id)
            ->where('amount', $payment->amount)
            ->where('status', '!=', 'paid')
            ->orderBy('due_date')
            ->first();
    }

    protected function processReconciliation(Payment $payment, Invoice $invoice)
    {
        // Check for discrepancies
        $discrepancy = $this->checkForDiscrepancies($payment, $invoice);

        if ($discrepancy) {
            $payment->forceFill([
                'reconciliation_status' => 'discrepancy',
                'reconciliation_notes' => $discrepancy
            ]);

            $this->logReconciliationHistory($payment, $invoice, 'discrepancy');
            return false;
        }

        // Process successful reconciliation
        $payment->forceFill([
            'invoice_id' => $invoice->id,
            'reconciliation_status' => 'reconciled',
            'reconciliation_notes' => null
        ]);

        $invoice->update([
            'status' => 'paid',
            //'amount' => $payment->amount,
            'paid_at' => now()
        ]);

        $this->logReconciliationHistory($payment, $invoice, 'reconciled');
        return true;
    }

    protected function checkForDiscrepancies(Payment $payment, Invoice $invoice)
    {
        if ($payment->amount != $invoice->total_amount) {
            return "Payment amount ({$payment->amount}) does not match invoice amount ({$invoice->total_amount})";
        }

        if ($payment->currency != $invoice->currency) {
            return "Payment currency ({$payment->currency}) does not match invoice currency ({$invoice->currency})";
        }

        return null;
    }

    protected function logReconciliationHistory(Payment $payment, ?Invoice $invoice, string $status)
    {
        $paymentInvoice = $payment->invoice;
        $customerPaket = $paymentInvoice->customer_paket;
        PaymentHistory::create([
            'payment_id' => $payment->id,
            'invoice_id' => $invoice?->id,
            'user_customer_id' => $payment->user_customer_id,
            'customer_name' => $payment->customer_name,
            'customer_address' => $payment->customer_address,
            'paket_name' => $customerPaket->paket->name,
            'net_customer_price' => $customerPaket->net_price,
            'period' => Carbon::parse($paymentInvoice->start_periode)->format('d M Y').' - '.Carbon::parse($paymentInvoice->end_periode)->format('d M Y'),
            'amount' => $payment->amount,
            //'currency' => $payment->currency,
            'payment_method' => $payment->payment_method,
            'transaction_id' => $payment->transaction_id,
            'teller_name' => $payment->teller,
            'bank_account' => $payment->bank ?? null,
            'status' => $status,
            'notes' => $payment->reconciliation_notes
        ]);
    }

    public function handleManualReconciliation(Payment $payment, Invoice $invoice)
    {
        return $this->processReconciliation($payment, $invoice);
    }
}
